// --- LÓGICA DE INICIO DE SESIÓN ---
const validUsers = [
    { username: 'i.cangana', password: 'itekne2025' },
    { username: 'r.carcamo', password: 'itekne2025' }
];

function handleLogin() {
    const username = document.getElementById('username').value.trim();
    const password = document.getElementById('password').value.trim();
    const errorMessage = document.getElementById('error-message');

    const foundUser = validUsers.find(user => user.username === username && user.password === password);

    if (foundUser) {
        sessionStorage.setItem('isLoggedIn', 'true');
        document.getElementById('login-overlay').classList.add('hidden');
        document.getElementById('app-wrapper').classList.remove('hidden');
    } else {
        errorMessage.classList.remove('hidden');
    }
}

function logout() {
    sessionStorage.removeItem('isLoggedIn');
    document.getElementById('login-overlay').classList.remove('hidden');
    document.getElementById('app-wrapper').classList.add('hidden');
}


// --- MODELO DE DATOS Y ESTADO ---
let appData = {
    services: [],
    clients: [],
    architects: [],
    config: {
        companyName: "Itekne Arquitectos",
        contactPhone: "+56994575900",
        contactEmail: "itekne.arq@gmail.com",
        ivaPercentage: 19
    }
};
let activeClientId = null;
let activeQuoteId = null;
let ivaAdded = false;

// --- LÓGICA DE NAVEGACIÓN ---
function switchView(viewName) {
    ['cotizaciones', 'historial', 'configuracion', 'estadisticas'].forEach(view => {
        document.getElementById(`view-${view}`).classList.add('hidden');
        document.getElementById(`nav-${view}`).classList.remove('active');
    });
    document.getElementById(`view-${viewName}`).classList.remove('hidden');
    document.getElementById(`nav-${viewName}`).classList.add('active');
    
    if (viewName === 'historial') renderHistoryView();
    if (viewName === 'configuracion') loadConfigIntoForm();
    if (viewName === 'estadisticas') renderStatisticsView();
}

// --- GESTIÓN DE CLIENTES ---
const clientModal = document.getElementById('client-modal');
function openClientModal(clientId = null) {
    const modalTitle = document.getElementById('client-modal-title');
    const idInput = document.getElementById('modal-client-id');
    const nameInput = document.getElementById('modal-client-name');
    const runInput = document.getElementById('modal-client-run');
    const phoneInput = document.getElementById('modal-client-phone');
    const emailInput = document.getElementById('modal-client-email');
    const addressInput = document.getElementById('modal-client-address');

    if (clientId) {
        const client = appData.clients.find(c => c.id === clientId);
        if (client) {
            modalTitle.textContent = "Editar Cliente";
            idInput.value = client.id;
            nameInput.value = client.name;
            runInput.value = client.run || '';
            phoneInput.value = client.phone || '';
            emailInput.value = client.email || '';
            addressInput.value = client.address || '';
        }
    } else {
        modalTitle.textContent = "Nuevo Cliente";
        idInput.value = '';
        nameInput.value = '';
        runInput.value = '';
        phoneInput.value = '';
        emailInput.value = '';
        addressInput.value = '';
    }
    clientModal.style.display = "flex";
}

function closeClientModal() { clientModal.style.display = "none"; }

function saveClientData() {
    const clientId = document.getElementById('modal-client-id').value;
    const clientData = {
        id: clientId || `client_${Date.now()}`,
        name: document.getElementById('modal-client-name').value.trim(),
        run: document.getElementById('modal-client-run').value,
        phone: document.getElementById('modal-client-phone').value,
        email: document.getElementById('modal-client-email').value,
        address: document.getElementById('modal-client-address').value,
    };
    if (!clientData.name) {
        showNotification("El nombre del cliente es obligatorio.", true);
        return;
    }

    let isNewClient = true;
    const clientIndex = appData.clients.findIndex(c => c.id === clientData.id);
    if (clientIndex > -1) {
        appData.clients[clientIndex] = { ...appData.clients[clientIndex], ...clientData };
        isNewClient = false;
    } else {
        clientData.quotes = [];
        appData.clients.push(clientData);
    }
    
    loadClient(clientData.id);
    saveDataToStorage();
    closeClientModal();
    renderHistoryView();
    showNotification(isNewClient ? `Cliente "${clientData.name}" creado` : `Cliente "${clientData.name}" actualizado`);
}

function deleteClient(clientId) {
    const client = appData.clients.find(c => c.id === clientId);
    if (!client) return;

    showConfirm(`¿Seguro que quieres eliminar a "${client.name}" y todas sus cotizaciones? Esta acción es irreversible.`, () => {
        const clientIndex = appData.clients.findIndex(c => c.id === clientId);
        if (clientIndex > -1) {
            appData.clients.splice(clientIndex, 1);
            saveDataToStorage();
            renderHistoryView();
            
            const quoteListTitle = document.getElementById('quote-list-title');
            if (quoteListTitle.textContent.includes(client.name)) {
                quoteListTitle.textContent = 'Cotizaciones';
                document.getElementById('quote-history-list').innerHTML = '';
            }
            showNotification(`Cliente "${client.name}" eliminado.`);
        }
    });
}


function loadClient(clientId) {
    const client = appData.clients.find(c => c.id === clientId);
    if (!client) return;

    activeClientId = clientId;
    document.getElementById('display-client-name').textContent = client.name;
    document.getElementById('display-client-run').textContent = client.run || '';
    document.getElementById('display-client-phone').textContent = client.phone || '';
    document.getElementById('display-client-email').textContent = client.email || '';
    document.getElementById('display-client-address').textContent = client.address || '';
    
    document.getElementById('display-quote-date').textContent = new Date().toLocaleDateString('es-CL');
    document.getElementById('display-quote-number').textContent = 'Nueva';
    
    clearQuoteTable();
}

function startNewQuoteForClient(clientId) {
    loadClient(clientId);
    switchView('cotizaciones');
}

// --- GESTIÓN DE SERVICIOS MAESTROS ---
const servicesModal = document.getElementById('services-modal');
function openServicesModal() { servicesModal.style.display = "flex"; }
function closeServicesModal() { servicesModal.style.display = "none"; }

function renderMasterServicesList() {
    const listElement = document.getElementById('master-services-list');
    listElement.innerHTML = appData.services.length === 0 ? '<li>No hay servicios.</li>' : '';
    appData.services.forEach((s, i) => {
        listElement.innerHTML += `<li class="master-service-item"><span>${s.description.replace(/\n/g, '<br>')}</span><span class="price">${formatCurrency(s.price)}</span><div><button onclick="editMasterService(${i})">✎</button><button onclick="deleteMasterService(${i})">✖</button></div></li>`;
    });
    populateServicesDropdown();
}

function addOrUpdateMasterService() {
    const desc = document.getElementById('master-service-description').value.trim();
    const price = parseFloat(document.getElementById('master-service-price').value);
    const index = document.getElementById('service-edit-index').value;
    if (!desc || isNaN(price)) {
        showNotification('Descripción y precio son requeridos.', true);
        return;
    }
    
    if (index) appData.services[index] = { description: desc, price };
    else appData.services.push({ description: desc, price });
    
    saveDataToStorage();
    renderMasterServicesList();
    document.getElementById('master-service-description').value = '';
    document.getElementById('master-service-price').value = '';
    document.getElementById('service-edit-index').value = '';
}

function editMasterService(index) {
    const service = appData.services[index];
    document.getElementById('master-service-description').value = service.description;
    document.getElementById('master-service-price').value = service.price;
    document.getElementById('service-edit-index').value = index;
    document.getElementById('service-form-btn').textContent = 'Actualizar Servicio';
    openServicesModal();
}

function deleteMasterService(index) {
    showConfirm('¿Está seguro de que desea eliminar este servicio de la lista maestra?', () => {
        appData.services.splice(index, 1);
        saveDataToStorage();
        renderMasterServicesList();
        showNotification("Servicio eliminado.");
    });
}

function populateServicesDropdown() {
    const select = document.getElementById('service-select');
    select.innerHTML = '<option value="">-- Seleccione un servicio --</option>';
    appData.services.forEach((s, i) => {
        select.innerHTML += `<option value="${i}">${s.description.split('\n')[0]}... (${formatCurrency(s.price)})</option>`;
    });
}

// --- LÓGICA DE COTIZACIÓN ---
function saveQuote() {
    if (!activeClientId) {
        showNotification("Seleccione o cree un cliente antes de guardar.", true);
        return;
    }
    const client = appData.clients.find(c => c.id === activeClientId);
    if (!client) return;

    const items = [];
    document.querySelectorAll('#services-table tbody tr').forEach(row => {
        const description = row.querySelector('.service-description').value.trim();
        const price = parseFloat(row.querySelector('.service-price').value);
        if (description && !isNaN(price)) items.push({ description, price });
    });
    if (items.length === 0) {
        showNotification("Añada al menos un servicio a la cotización.", true);
        return;
    }

    const architectId = document.getElementById('architect-select').value;
    if (!architectId) {
        showNotification("Seleccione un autor para la cotización.", true);
        return;
    }
    
    const quoteNumber = (client.quotes.length || 0) + 1;
    const newQuote = {
        id: `quote_${Date.now()}`,
        number: quoteNumber,
        date: new Date().toISOString(),
        items: items,
        iva: ivaAdded,
        architectId: architectId,
        status: 'Pendiente'
    };

    client.quotes.push(newQuote);
    saveDataToStorage();
    showNotification(`Cotización #${quoteNumber} guardada para ${client.name}`);
    renderHistoryView();
    loadQuote(client.id, newQuote.id);
}

function loadQuote(clientId, quoteId) {
    const client = appData.clients.find(c => c.id === clientId);
    const quote = client?.quotes.find(q => q.id === quoteId);
    if (!quote) return;
    
    loadClient(clientId);
    activeQuoteId = quoteId;
    
    document.getElementById('display-quote-date').textContent = new Date(quote.date).toLocaleDateString('es-CL');
    document.getElementById('display-quote-number').textContent = quote.number;

    const tableBody = document.querySelector("#services-table tbody");
    tableBody.innerHTML = '';
    quote.items.forEach(item => addServiceRow(item.description, item.price));
    
    document.getElementById('architect-select').value = quote.architectId || '';
    updateQuoteAuthorDisplay();

    ivaAdded = quote.iva;
    updateTotals();
    toggleIVA(true);
    switchView('cotizaciones');
}

function deleteQuote(clientId, quoteId) {
    showConfirm('¿Está seguro de que desea eliminar esta cotización de forma permanente?', () => {
        const clientIndex = appData.clients.findIndex(c => c.id === clientId);
        if (clientIndex > -1) {
            const quoteIndex = appData.clients[clientIndex].quotes.findIndex(q => q.id === quoteId);
            if (quoteIndex > -1) {
                appData.clients[clientIndex].quotes.splice(quoteIndex, 1);
                saveDataToStorage();
                renderHistoryView();
                renderQuoteList(clientId);
                showNotification("Cotización eliminada con éxito.");
            }
        }
    });
}

function clearQuote() {
    showConfirm('¿Limpiar la cotización actual? Los cambios no guardados se perderán.', () => {
        activeClientId = null;
        activeQuoteId = null;
        document.getElementById('display-client-name').textContent = 'Ningún cliente cargado';
        ['run', 'phone', 'email', 'address', 'quote-date', 'quote-number'].forEach(f => document.getElementById(`display-${f}`).textContent = '');
        clearQuoteTable();
    });
}

function clearQuoteTable() {
    document.querySelector("#services-table tbody").innerHTML = '';
    addServiceRow();
    document.getElementById('architect-select').value = '';
    updateQuoteAuthorDisplay();
    ivaAdded = false;
    updateTotals();
    toggleIVA(true);
}

function addServiceRow(desc = '', price = 0) {
    const tableBody = document.querySelector("#services-table tbody");
    const newRow = tableBody.insertRow();
    newRow.innerHTML = `<td><textarea class="service-description" oninput="updateTotals()">${desc}</textarea></td><td><input type="number" class="service-price" value="${price}" oninput="updateTotals()"></td><td><button class="delete-row-btn" onclick="this.closest('tr').remove(); updateTotals();">X</button></td>`;
    updateTotals();
}

function addSelectedServiceToQuote() {
    const index = document.getElementById('service-select').value;
    if (index === "") return;
    const service = appData.services[index];
    addServiceRow(service.description, service.price);
}

function updateTotals() {
    let subtotal = 0;
    document.querySelectorAll(".service-price").forEach(i => subtotal += parseFloat(i.value) || 0);
    document.getElementById("subtotal").textContent = formatCurrency(subtotal);
    let iva = ivaAdded ? subtotal * (appData.config.ivaPercentage / 100) : 0;
    document.getElementById("iva-container").classList.toggle('hidden', !ivaAdded);
    document.getElementById("iva-amount").textContent = formatCurrency(iva);
    document.getElementById("total").textContent = formatCurrency(subtotal + iva);
}
    
function toggleIVA(forceUpdate = false) {
    if (!forceUpdate) ivaAdded = !ivaAdded;
    document.getElementById("add-iva-btn").textContent = ivaAdded ? "Quitar IVA" : `Agregar IVA (${appData.config.ivaPercentage}%)`;
    updateTotals();
}

// --- VISTA DE HISTORIAL ---
function renderHistoryView() {
    const clientList = document.getElementById('client-history-list');
    const searchTerm = document.getElementById('client-search-input').value.toLowerCase();
    clientList.innerHTML = '';
    
    const filteredClients = appData.clients.filter(client => client.name.toLowerCase().includes(searchTerm));

    filteredClients.forEach(client => {
        const item = document.createElement('li');
        item.className = 'client-list-item';
        
        item.innerHTML = `
            <div class="client-info" onclick="renderQuoteList('${client.id}')">
                <div class="name">${client.name}</div>
                <div class="count">${client.quotes.length} cotizacion(es)</div>
            </div>
            <div class="client-actions">
                <button class="action-btn" style="padding: 5px 10px; background-color: #3498db; width: 100%;" onclick="startNewQuoteForClient('${client.id}')">Cotizar</button>
                <button class="action-btn" style="padding: 5px 10px; background-color: #f1c40f; width: 100%;" onclick="openClientModal('${client.id}')">Editar</button>
                <button class="action-btn" style="padding: 5px 10px; background-color: #e74c3c;" onclick="deleteClient('${client.id}')">🗑️</button>
            </div>
        `;
        clientList.appendChild(item);
    });
}

function renderQuoteList(clientId) {
    const allClientItems = document.querySelectorAll('.client-list-item');
    allClientItems.forEach(item => item.classList.remove('selected'));
    const clientItem = Array.from(allClientItems).find(item => item.innerHTML.includes(`'${clientId}'`));
    if (clientItem) clientItem.classList.add('selected');

    const client = appData.clients.find(c => c.id === clientId);
    const quoteList = document.getElementById('quote-history-list');
    document.getElementById('quote-list-title').textContent = `Cotizaciones de ${client.name}`;
    quoteList.innerHTML = '';
    if (!client || client.quotes.length === 0) {
        quoteList.innerHTML = '<li>No hay cotizaciones para este cliente.</li>';
        return;
    }
    client.quotes.sort((a, b) => a.number - b.number).forEach(quote => {
        const item = document.createElement('li');
        item.className = 'quote-list-item';
        const quoteDate = new Date(quote.date).toLocaleDateString('es-CL');
        const status = quote.status || 'Pendiente';
        item.innerHTML = `
            <div>
                <span><strong>#${quote.number}</strong> - ${quoteDate}</span>
                <span class="quote-status status-${status}">${status}</span>
            </div>
            <div class="quote-item-actions">
                <select onchange="changeQuoteStatus('${clientId}', '${quote.id}', this.value)">
                    <option value="Pendiente" ${status === 'Pendiente' ? 'selected' : ''}>Pendiente</option>
                    <option value="Aprobada" ${status === 'Aprobada' ? 'selected' : ''}>Aprobada</option>
                    <option value="Rechazada" ${status === 'Rechazada' ? 'selected' : ''}>Rechazada</option>
                </select>
                <button class="action-btn" style="padding: 5px 10px;" onclick="loadQuote('${clientId}', '${quote.id}')">Editar</button>
                <button class="action-btn" style="padding: 5px 10px; background-color: #e74c3c;" onclick="deleteQuote('${clientId}', '${quote.id}')">🗑️</button>
            </div>
        `;
        quoteList.appendChild(item);
    });
}

function changeQuoteStatus(clientId, quoteId, newStatus) {
    const client = appData.clients.find(c => c.id === clientId);
    const quote = client?.quotes.find(q => q.id === quoteId);
    if (quote) {
        quote.status = newStatus;
        saveDataToStorage();
        renderQuoteList(clientId);
        showNotification(`Estado de la cotización #${quote.number} actualizado a "${newStatus}".`);
    }
}

// --- VISTA DE ESTADÍSTICAS ---
function renderStatisticsView() {
    const statsTableBody = document.querySelector('#stats-table tbody');
    statsTableBody.innerHTML = '';
    appData.clients.forEach(client => {
        const row = statsTableBody.insertRow();
        row.innerHTML = `
            <td>${client.name}</td>
            <td>${client.quotes.length}</td>
            <td><button class="action-btn" style="padding: 5px 10px;" onclick="viewClientQuotes('${client.id}')">Ver Cotizaciones</button></td>
        `;
    });
}

function viewClientQuotes(clientId) {
    switchView('historial');
    setTimeout(() => {
        renderQuoteList(clientId);
    }, 0);
}


// --- GESTIÓN DE CONFIGURACIÓN Y ARQUITECTOS ---
function saveConfig() {
    appData.config.companyName = document.getElementById('config-company-name').value;
    appData.config.contactPhone = document.getElementById('config-contact-phone').value;
    appData.config.contactEmail = document.getElementById('config-contact-email').value;
    appData.config.ivaPercentage = parseFloat(document.getElementById('config-iva-percentage').value);
    saveDataToStorage();
    applyConfig();
    showNotification("Configuración guardada con éxito.");
}

function loadConfigIntoForm() {
    document.getElementById('config-company-name').value = appData.config.companyName;
    document.getElementById('config-contact-phone').value = appData.config.contactPhone;
    document.getElementById('config-contact-email').value = appData.config.contactEmail;
    document.getElementById('config-iva-percentage').value = appData.config.ivaPercentage;
    renderArchitectsList();
}

function applyConfig() {
    document.getElementById('company-name-display').textContent = appData.config.companyName;
    document.getElementById('contact-phone-display').textContent = appData.config.contactPhone;
    document.getElementById('contact-email-display').textContent = appData.config.contactEmail;
    document.getElementById('iva-label').textContent = `IVA (${appData.config.ivaPercentage}%):`;
    populateArchitectsDropdown();
    updateTotals();
}

function addArchitect() {
    const nameInput = document.getElementById('architect-name-input');
    const name = nameInput.value.trim();
    if (name) {
        appData.architects.push({ id: `arch_${Date.now()}`, name: name });
        saveDataToStorage();
        renderArchitectsList();
        populateArchitectsDropdown();
        nameInput.value = '';
    }
}

function deleteArchitect(index) {
    appData.architects.splice(index, 1);
    saveDataToStorage();
    renderArchitectsList();
    populateArchitectsDropdown();
}

function renderArchitectsList() {
    const container = document.getElementById('architects-list-container');
    container.innerHTML = '';
    appData.architects.forEach((arch, index) => {
        container.innerHTML += `<div class="architect-item"><span>${arch.name}</span><button class="delete-row-btn" onclick="deleteArchitect(${index})">X</button></div>`;
    });
}

function populateArchitectsDropdown() {
    const select = document.getElementById('architect-select');
    select.innerHTML = '<option value="">-- Seleccione un arquitecto --</option>';
    appData.architects.forEach(arch => {
        select.innerHTML += `<option value="${arch.id}">${arch.name}</option>`;
    });
}

function updateQuoteAuthorDisplay() {
    const architectId = document.getElementById('architect-select').value;
    const authorDisplay = document.getElementById('quote-author-display');
    const architect = appData.architects.find(a => a.id === architectId);
    authorDisplay.textContent = architect ? architect.name : 'Itekne arquitectos';
}

// --- UTILIDADES Y ALMACENAMIENTO ---
function formatCurrency(v) { return new Intl.NumberFormat('es-CL', { style: 'currency', currency: 'CLP' }).format(v); }

async function downloadPDF() {
    const downloadBtn = document.getElementById('download-pdf-btn');
    downloadBtn.textContent = 'Generando...';
    downloadBtn.disabled = true;

    const invoiceElement = document.getElementById('invoice');
    
    const elementsToHide = [
        invoiceElement.querySelector('.services-section'),
        ...invoiceElement.querySelectorAll('.delete-row-btn')
    ];

    elementsToHide.forEach(el => { if(el) el.style.visibility = 'hidden'; });

    try {
        const canvas = await html2canvas(invoiceElement, { 
            scale: 2,
            useCORS: true,
            allowTaint: true
        });
        const imgData = canvas.toDataURL('image/png');
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('p', 'mm', 'a4'); 
        const pdfWidth = pdf.internal.pageSize.getWidth();
        const pdfHeight = (canvas.height * pdfWidth) / canvas.width;

        pdf.addImage(imgData, 'PNG', 0, 0, pdfWidth, pdfHeight);

        const clientName = document.getElementById('display-client-name').textContent.trim();
        const finalClientName = (clientName && clientName !== 'Ningún cliente cargado') ? clientName : 'cliente';
        const fileName = `Cotizacion-${finalClientName.replace(/ /g, '_')}.pdf`;
        
        pdf.save(fileName);

    } catch (error) {
        console.error("Error al generar el PDF:", error);
        showNotification("Hubo un error al generar el PDF.", true);
    } finally {
        elementsToHide.forEach(el => { if(el) el.style.visibility = 'visible'; });
        downloadBtn.textContent = 'Descargar PDF';
        downloadBtn.disabled = false;
    }
}

const confirmModal = document.getElementById('confirm-modal');
let onConfirmCallback = null;
function showConfirm(text, callback) {
    document.getElementById('confirm-modal-text').textContent = text;
    onConfirmCallback = callback;
    confirmModal.style.display = "flex";
}
function hideConfirm() { confirmModal.style.display = "none"; }
document.getElementById('confirm-ok-btn').addEventListener('click', () => { if (onConfirmCallback) onConfirmCallback(); hideConfirm(); });
document.getElementById('confirm-cancel-btn').addEventListener('click', hideConfirm);

const notificationPopup = document.getElementById('notification-popup');
const notificationText = document.getElementById('notification-text');
let notificationTimeout;

function showNotification(message, isError = false) {
    clearTimeout(notificationTimeout);
    notificationText.textContent = message;
    notificationPopup.classList.toggle('error', isError);
    notificationPopup.classList.add('show');
    notificationTimeout = setTimeout(() => {
        notificationPopup.classList.remove('show');
    }, 3000);
}

function saveDataToStorage() {
    localStorage.setItem('itecneAppData', JSON.stringify(appData));
}

function loadDataFromStorage() {
    const storedData = localStorage.getItem('itecneAppData');
    if (storedData) {
        const parsedData = JSON.parse(storedData);
        appData = { ...appData, ...parsedData };
    }
}

function exportToCSV(data, filename) {
    if (data.length === 0) {
        showNotification("No hay datos para exportar.", true);
        return;
    }
    const headers = Object.keys(data[0]);
    const csvRows = [headers.join(',')];
    
    data.forEach(row => {
        const values = headers.map(header => {
            const escaped = ('' + row[header]).replace(/"/g, '""');
            return `"${escaped}"`;
        });
        csvRows.push(values.join(','));
    });

    const blob = new Blob([csvRows.join('\n')], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function exportClientsToCSV() {
    const clientsToExport = appData.clients.map(({ id, quotes, ...rest }) => rest);
    exportToCSV(clientsToExport, 'clientes_itekne.csv');
}

function exportServicesToCSV() {
    exportToCSV(appData.services, 'servicios_itekne.csv');
}
    
window.onload = () => {
    // Comprueba si el usuario ya ha iniciado sesión en esta sesión del navegador
    if (sessionStorage.getItem('isLoggedIn') === 'true') {
        document.getElementById('login-overlay').classList.add('hidden');
        document.getElementById('app-wrapper').classList.remove('hidden');
    }

    // Configura los listeners del formulario de login
    document.getElementById('login-btn').addEventListener('click', handleLogin);
    document.getElementById('password').addEventListener('keyup', (event) => {
        if (event.key === 'Enter') {
            handleLogin();
        }
    });

    // Carga el resto de la aplicación
    loadDataFromStorage();
    applyConfig();
    renderMasterServicesList();
    clearQuoteTable();

    document.getElementById('architect-select').addEventListener('change', updateQuoteAuthorDisplay);

    window.onclick = function(event) {
        if (event.target == clientModal) closeClientModal();
        if (event.target == servicesModal) closeServicesModal();
        if (event.target == confirmModal) hideConfirm();
    }
};

function logout() {
    sessionStorage.removeItem('isLoggedIn');
    document.getElementById('login-overlay').classList.remove('hidden');
    document.getElementById('app-wrapper').classList.add('hidden');
    // Limpia los campos del formulario de login al cerrar sesión
    document.getElementById('username').value = '';
    document.getElementById('password').value = '';
    document.getElementById('error-message').classList.add('hidden');
}
